<?php
/**
 * Ninja functions and definitions
 *
 * @package ninja
 */

 /**
 * Set the content width based on the theme's design and stylesheet.
 * Used to set the width of images and content. Should be equal to the width the theme is designed for, generally via the style.css stylesheet.
 */
if ( ! isset( $content_width ) ) { $content_width = 648; /* pixels */ }

/**
 * Set the content width for full width pages with no sidebar.
 */
function ninja_content_width() {
  if ( is_page_template( 'page-fullwidth.php' ) ) {
    global $content_width;
    $content_width = 1008; /* pixels */
  }
}
add_action( 'template_redirect', 'ninja_content_width' );

/**
 * Set container based on template
 */
function set_ninja_container(){
    $container = [];
    $templates = [];

    $templates = ['page-fullwidth.php', 'page-home.php', 'page-with-sidebar.php'];

    if ( is_page_template($templates) || is_product() ) {
        $container['container_type'] = 'container-fluid';
    }else{
        $container['container_type'] = 'container';
    }

    return $container;
}

function ninja_container(){
    $container = set_ninja_container();
    $ninja_container = $container['container_type'];
    return $ninja_container;
}

if ( ! function_exists( 'ninja_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function ninja_setup() {
    // Make theme available for translation. Translations can be filed in the /languages/ directory.
    //load_theme_textdomain( 'ninja', get_template_directory() . '/languages' );
    // Enable support for Post Thumbnails on posts and pages.
    add_theme_support( 'post-thumbnails' );
    // This theme uses wp_nav_menu() in one location.
    register_nav_menus( array(
        'primary'      => esc_html__( 'Primary Menu', 'ninja' ),
        'social-menu'  => esc_html__( 'Social Menu', 'ninja' )
    ) );
    // Enable support for HTML5 markup.
    add_theme_support( 'html5', array(
        'search-form',
        'gallery',
        'caption'
    ) );
    // Let WordPress manage the document title in the document head.
    add_theme_support( 'title-tag' );
    // Enable support for WooCommerce.
    add_theme_support( 'woocommerce' );
    // Enable support for custom header
    register_default_headers( array(
        'default-image' => array(
            'url'           => get_stylesheet_directory_uri() . '/images/logo2.jpg',
            'thumbnail_url' => get_stylesheet_directory_uri() . '/images/logo2.jpg',
            'description'   => __( 'Default Logo', 'ninja' )
        ),
    ) );
    $header_args = array(
        'default-image'         => '',
        'default-text-color'    => '000',
        'width'                 => 220,
        'height'                => 125,
        'flex-height'           => true,
        'flex-width'            => true,
    );
    add_theme_support( 'custom-header', $header_args );

}
endif; // ninja_setup
add_action( 'after_setup_theme', 'ninja_setup' );

/**
 * Register widgetized area and update sidebar with default widgets.
 */
function ninja_widgets_init() {

    register_sidebar( array(
        'id'            => 'sidebar-widget',
        'name'          => esc_html__( 'Sidebar', 'ninja' ),
        'description'   => esc_html__( 'The default sidebar widget', 'ninja' ),
        'before_widget' => '<aside id="%1$s" class="widget %2$s">',
        'after_widget'  => '</aside>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'id'            => 'footer-widget-1',
        'name'          => esc_html__( 'Footer Area 1', 'ninja' ),
        'description'   => esc_html__( 'The footer widget area 1', 'ninja' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'id'            => 'footer-widget-2',
        'name'          => esc_html__( 'Footer Area 2', 'ninja' ),
        'description'   => esc_html__( 'The footer widget area 2', 'ninja' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'id'            => 'footer-widget-3',
        'name'          => esc_html__( 'Footer Area 3', 'ninja' ),
        'description'   => esc_html__( 'The footer widget area 3', 'ninja' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );

     register_sidebar( array(
        'id'            => 'floating-widget',
        'name'          => esc_html__( 'Floating Widget', 'ninja' ),
        'description'   => esc_html__( 'Floating Widget', 'ninja' ),
        'before_widget' => '<div id="%1$s" class="floating widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
     ) );

    register_widget( 'ninja_social_widget' );
}
add_action( 'widgets_init', 'ninja_widgets_init' );

/**
 * Theme Widgets
 */
require_once(get_template_directory() . '/inc/widget-social.php');

/**
 * Unregister all unnecessary widgets 
 */
function ninja_unregister_default_widgets() {
    unregister_widget('WP_Widget_Pages');
    unregister_widget('WP_Widget_Calendar');
    unregister_widget('WP_Widget_Links');
    unregister_widget('WP_Widget_Meta');
    unregister_widget('WP_Widget_Recent_Comments');
    unregister_widget('WP_Widget_Recent_Posts');
    unregister_widget('WP_Widget_RSS');
    unregister_widget('WP_Widget_Tag_Cloud');

    // WOO WIDGETS
    unregister_widget( 'WC_Widget_Layered_Nav' );
    unregister_widget( 'WC_Widget_Layered_Nav_Filters' );
    unregister_widget( 'WC_Widget_Price_Filter' );
    unregister_widget( 'WC_Widget_Product_Tag_Cloud' );
    unregister_widget( 'WC_Widget_Recent_Reviews' );
    unregister_widget( 'WC_Widget_Recently_Viewed' );
    unregister_widget( 'WC_Widget_Top_Rated_Products' );
} 
add_action('widgets_init', 'ninja_unregister_default_widgets', 11);

/**
 * This function removes inline styles set by WordPress gallery.
 */
function ninja_remove_gallery_css( $css ) {
  return preg_replace( "#<style type='text/css'>(.*?)</style>#s", '', $css );
}
add_filter( 'gallery_style', 'ninja_remove_gallery_css' );

/**
 * Enqueue scripts and styles.
 */
function ninja_scripts() {

  // Bootstrap default CSS
  wp_enqueue_style( 'ninja-bootstrap', get_template_directory_uri() . '/inc/css/bootstrap.min.css' );

  // Font Awesome
  wp_enqueue_style( 'ninja-icons', '//use.fontawesome.com/releases/v5.1.0/css/all.css' );

  // Google Fonts
  wp_enqueue_style( 'ninja-fonts', '//fonts.googleapis.com/css?family=Ubuntu:400,400i,700');

  // Main theme stylesheet
  wp_enqueue_style( 'ninja-style', get_stylesheet_uri() );

  // Load jquery
  wp_enqueue_script('jquery');

  // Modernizr for better HTML5 and CSS3 support
  wp_enqueue_script('ninja-modernizr', get_template_directory_uri().'/inc/js/modernizr.min.js', array('jquery') );

  // Bootstrap default JS
  wp_enqueue_script('ninja-bootstrapjs', get_template_directory_uri().'/inc/js/bootstrap.min.js', array('jquery') );

  // Main theme related functions
  wp_enqueue_script( 'ninja-functions', get_template_directory_uri() . '/inc/js/functions.min.js', array('jquery') );

  // Smooth scroll script
  wp_enqueue_script( 'smooth-scroll-script', get_template_directory_uri() . '/inc/js/smooth.scroll.js', array('jquery') );

  // Responsive Grids
  wp_enqueue_script( 'responsive-grids', get_template_directory_uri() . '/js/grids.min.js', array('jquery') );

  // Owl carousel
  wp_enqueue_style( 'OwlCarousel2-style', get_template_directory_uri().'/OwlCarousel2/owl.carousel.min.css' );
  wp_enqueue_style( 'OwlCarousel2-theme', get_template_directory_uri().'/OwlCarousel2/owl.theme.default.min.css' );
  wp_enqueue_script( 'OwlCarousel2-script', get_template_directory_uri() . '/OwlCarousel2/owl.carousel.min.js', array('jquery') );

  // Custom style
  wp_enqueue_style( 'ninja-custom-style', get_template_directory_uri().'/custom.css' );

  // Custom scripts
  if ( is_page_template('page-thankyou.php') ) {
     wp_enqueue_script( 'thank-you-script', get_template_directory_uri() . '/js/tqcountdown.js', array('jquery') ,'1.', true);
  }
  wp_enqueue_script( 'product-carousel-script', get_template_directory_uri() . '/js/product.carousel.js', array('jquery') );
  wp_enqueue_script( 'custom-script', get_template_directory_uri() . '/js/custom.js', array('jquery') );

}
add_action( 'wp_enqueue_scripts', 'ninja_scripts' );

/*function woocommerce_styling(){
    wp_enqueue_style( 'woocommerce-style', get_template_directory_uri().'/woocommerce/woocommerce.css' );
}*/

if ( in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
    /*add_action( 'wp_enqueue_scripts', 'woocommerce_styling' );*/
    require get_template_directory() . '/woocommerce/woo-functions.php';
}

/**
 * Extra functions for this theme.
 */
require get_template_directory() . '/inc/navwalker.php'; // CUSTOM NAV WALKER
require get_template_directory() . '/inc/navigation.php'; // NAVIGATION RELATED FUNCTIONS
require get_template_directory() . '/inc/extras.php'; // THEME RELATED FUNCTIONS

/**
 * Default Excerpt Length
 */
function ninja_excerpt_length( $length ) {
    return 25;
}
add_filter( 'excerpt_length', 'ninja_excerpt_length' );

/**
 * Shortcodes in excerpt and widgets
 */
add_filter('the_excerpt', 'do_shortcode');
add_filter('widget_text','do_shortcode');

/**
 * Allow shortcodes inside Contact Form 7 forms
 */
function shortcode_wpcf7_form( $form ) {
    $form = do_shortcode( $form );
    return $form;
}
add_filter( 'wpcf7_form_elements', 'shortcode_wpcf7_form' );

/**
 * Create layout for ninja breadcrumbs
 */
function ninja_breadcrumbs(){

    if ( function_exists('bcn_display') ){
        $output = '';
        $output .= '<!--start breadcrumbs--><div class="breadcrumbs">';

        $container = set_ninja_container();
        $ninja_container = $container['container_type'];
        $breadcrumbs = bcn_display(true, true, false);

        if ( $ninja_container == 'container-fluid' ){
            $output .= '<div class="container"><div class="row"><div class="col-sm-12">';
            $output .= $breadcrumbs;
            $output .= '</div></div></div>';
        }else{
            $output .= $breadcrumbs;
        }
        
        $output .= '</div><!--end breadcrumbs-->';
        echo $output;
    }

}
add_action( 'ninja_breadcrumbs', 'ninja_breadcrumbs');

/**
 * Set pagenavi plugin as default navigation
 */
function ninja_navigation(){
    if ( function_exists('wp_pagenavi') ) { wp_pagenavi(); }
}
add_action( 'ninja_navigation', 'ninja_navigation');

/**
 * Remove archive type from title
 */
add_filter( 'get_the_archive_title', 'remove_prefix_archive_title' );
function remove_prefix_archive_title ($title) {
    if ( is_category() ) {
        $title = single_cat_title( '', false );
    } elseif ( is_tag() ) {
        $title = single_tag_title( '', false );
    } elseif ( is_author() ) {
        $title = '<span class="vcard">' . get_the_author() . '</span>' ;
    } elseif ( is_tax() ){
        $title = single_term_title( '', false );
    } elseif ( is_post_type_archive() ){
        $title = post_type_archive_title( '', false );
    }
    return $title;
}

/* Hide WP version strings from scripts and styles
* @return {string} $src
* @filter script_loader_src
* @filter style_loader_src
*/
function ninja_remove_wp_version_strings( $src ) {
    global $wp_version;
    parse_str(parse_url($src, PHP_URL_QUERY), $query);
    if ( !empty($query['ver']) && $query['ver'] === $wp_version ) {
        $src = remove_query_arg('ver', $src);
    }
    return $src;
}
add_filter( 'script_loader_src', 'ninja_remove_wp_version_strings' );
add_filter( 'style_loader_src', 'ninja_remove_wp_version_strings' );

/* Hide WP version strings from generator meta tag */
function ninja_remove_version() {
    return '';
}
add_filter('the_generator', 'ninja_remove_version');

// Disable access to author page
add_action('template_redirect', 'ninja_disable_author_page');
function ninja_disable_author_page() {
    global $wp_query;
    if ( is_author() ) {
        $wp_query->set_404();
        status_header(404);
        // Redirect to homepage
        wp_redirect(get_option('home'));
    }
}
add_action( 'template_redirect', 'ninja_disable_author_page' );

/**
 * Font Awesome shortcodes
 */
add_shortcode( 'fa', 'shortcode_icon' );
function shortcode_icon( $atts ){
    $n = shortcode_atts( array(
        'xclass'    => '',
        'icon'      => '',
        'size'      => '',
        'brand'     => 0,
        'regular'   => 0,
        'fw'        => 0,
    ), $atts );

    $xclass     = $n['xclass'];
    $icon       = $n['icon'];
    $size       = $n['size'];
    $brand      = $n['brand'];
    $regular    = $n['regular'];
    $fw         = $n['fw'];

    // Exit function if no icon class provided
    if( empty( $icon ) ){ return; }

    // Determine if brand style or regular or solid
    if ( $brand === 'false' ) { $brand = false; }
    $brand = (bool) $brand;

    if ( $regular === 'false' ) { $regular = false; }
    $regular = (bool) $regular;

    if ( $brand === true ){
        $class_array[] = 'fab';
    } elseif ( $regular === true ){
        $class_array[] = 'far';
    } else {
        $class_array[] = 'fas';
    }

    // Replace any empty space with dash & add icon class to array
    $icon = str_replace(' ', '-', $icon);
    $class_array[] = 'fa-' . $icon;

    // Determine if fixed width enabled
    if ( $fw === 'false' ) { $fw = false; }
    $fw = (bool) $fw;

    if ( $fw === true ){
        $class_array[] = 'fa-fw';
    }

    // Icon Size
    $size_opts = array('xs', 'sm', 'lg', '2x', '3x', '4x', '5x', '6x', '7x', '8x', '9x', '10x');
    $size_result = count(array_filter($size_opts, create_function('$e','return strstr("'.$size.'", $e);')))>0;
    if( !empty( $size ) && $size_result !== false ){
        $class_array[] = 'fa-' . $size;
    }

    // Add extra classes
    if ( !empty( $xclass ) ){
        $class_array[] = $xclass;
    }

    return '<i class="' . implode( ' ', $class_array ) . '"></i>';
}

/**
 * Change CF7 Submit Input to Button
 */
remove_action( 'wpcf7_init', 'wpcf7_add_form_tag_submit', 0 );
add_action( 'wpcf7_init', 'wpcf7_add_form_tag_submit_button', 0 );
function wpcf7_add_form_tag_submit_button() { 
    wpcf7_add_form_tag( 'submit', 'wpcf7_submit_button_form_tag_handler' ); 
}
function wpcf7_submit_button_form_tag_handler( $tag ) {
    $class = wpcf7_form_controls_class( $tag->type );

    $atts = array();

    $atts['class'] = $tag->get_class_option( $class ) . ' btn btn-default';
    $atts['id'] = $tag->get_id_option();
    $atts['tabindex'] = $tag->get_option( 'tabindex', 'signed_int', true );

    $value = isset( $tag->values[0] ) ? $tag->values[0] : '';

    if ( empty( $value ) ) {
        $value = __( 'Send', 'contact-form-7' );
    }

    $atts['type'] = 'submit';
    $atts['data-text'] = $value;
    $atts = wpcf7_format_atts( $atts );
    $html = sprintf(    '<button %1$s>%2$s</button>', 
                        $atts, 
                        $value );
    return $html;
}

/**
 * Grab all Toolset Types Custom Fields by Group ID or Group Slug
 */
function types_render_group($target_group) {
    global $post;
    require_once WPCF_EMBEDDED_INC_ABSPATH . '/fields.php';
     
    //by known group_id
    if(is_numeric($target_group)){
        return wpcf_admin_fields_get_fields_by_group($target_group);
    }
    //by known slug
    else{
        $groups = wpcf_admin_fields_get_groups();
        foreach($groups AS $group){
            if($group['slug'] == $target_group){
                return wpcf_admin_fields_get_fields_by_group($group['id']);
            }
        }
    }
    //check yourself, dude
    return false;
}

/*Background Image Shortcode*/
function nbg_img( $atts, $content = null ){

    $n = shortcode_atts( array(
        'xclass'    => '',
        'link'      => '',
        'tag'       => '',
        'id'        => '',
        'img_id'    => '',
        'color'     => '',
        'bg_color'  => '',
        'repeat'    => 'no-repeat',
        'position'  => 'center',
        'size'      => 'cover',
        'gradient'  => '',
        'parallax'  => false,
    ), $atts );

    $class_array        = array();
    $class_array[]      = $n['xclass'];
    $link               = $n['link'];
    $tag                = $n['tag'];
    $id                 = $n['id'];
    $img_id             = $n['img_id'];
    $color              = $n['color'];
    $bg_color           = $n['bg_color'];
    $repeat             = $n['repeat'];
    $position           = $n['position'];
    $size               = $n['size'];
    $gradient           = $n['gradient'];
    $parallax           = $n['parallax'];
    $style_atts         = array();

    $tag_options = array('div','section');
    $result = count(array_filter($tag_options, create_function('$e','return strstr("'.$tag.'", $e);')))>0;

    if( empty( $tag ) && $result === false ){
        $tag = 'div';
    }

    if ( $parallax === false ){
        $class_array[] = 'no-parallax-effect';
    } else {
        $class_array[] = 'parallax-effect';
    }

    if ( !empty( $img_id ) ){
        $bg_image_url = wp_get_attachment_url( $img_id );
    }

    if ( !empty( $link ) ){
        $bg_image_url = $link;
    }

    if ( !empty( $bg_image_url ) ){

        if ( !empty( $gradient ) ){
            $gradient = 'linear-gradient('.$gradient.'), ';
        }

        $style_atts[] = 'background-image: '.$gradient.'url('.$bg_image_url.')';
    }

    if ( !empty( $bg_color ) ){
        $style_atts[] = 'background-color: '.$bg_color;
    }

    if ( !empty( $repeat ) ){
        $style_atts[] = 'background-repeat: '.$repeat;
    }

    if ( !empty( $position ) ){
        $style_atts[] = 'background-position: '.$position;
    }

    if ( !empty( $size ) ){
        $style_atts[] = 'background-size: '.$size;
    }

    if ( !empty( $id ) ){
        $id = 'id="'.$id.'"';
    }

    if ( !empty( $color ) ){
        $style_atts[] = 'color: '.$color;
    }

    return '<'.$tag.' '.$id.' class="nbg-img ' . implode( ' ', $class_array ) . '" style="' . implode( '; ', $style_atts ) . '">'.do_shortcode($content).'</'.$tag.'>';
}
add_shortcode('nbg-img', 'nbg_img');

/**
 * Default page banner
 */
add_action( 'page_banner', 'page_banner');
function page_banner() {

    if ( is_front_page() || is_page_template(array('page-thankyou.php','page-home.php')) ) { return; }

    $output             = null;
    $title              = null;
    $page_title         = null;
    $banner_url         = null;

    $banner_url = wp_get_attachment_image_src( 58, 'full', false )[0];

    if ( is_search() ){
        $title = 'Search Results';
    } elseif ( is_404() ){
        $title = '404 Error';
    } elseif ( in_category('news-events') ){
        $title = 'News & Events';
    } elseif ( is_product() || is_shop() || is_product_category() ){
        $title = 'Products';
    } elseif ( is_archive() ){
        $title = get_the_archive_title();
    } else {
        $title = get_the_title();
    }

    $page_title = '<h2 class="page-title">'.$title.'</h2>';

    $output .=  '<div class="page-banner nvp30 nflex-vcenter" style="background-image: linear-gradient(rgba(0,0,0,.6), rgba(0,0,0,.6)), url('.$banner_url.');">'.
                '<div class="banner-inner container nflex-md nflex-vcenter xs-center sm-center">'.
                $page_title.
                '<div class="breadcrumbs">'.
                bcn_display(true, true, false).
                '</div><!-- .breadcrumbs -->'.
                '</div><!-- .banner-inner -->'.
                '</div><!-- .page-banner -->';
    echo $output;
}

/**
 * Coming soon layout shortocde
 */
function coming_soon_layout_shortcode() {
   ob_start();
   get_template_part('content', 'coming-soon');
   return ob_get_clean();   
} 
add_shortcode( 'coming_soon_layout', 'coming_soon_layout_shortcode' );